<?php
defined( 'ABSPATH' ) || exit;

class HRM_Portal {

    public static function init() {
         add_shortcode( 'hrm_login', [ __CLASS__, 'render_login_page' ] );
         add_shortcode( 'hrm_employee_portal', [ __CLASS__, 'render_employee_portal' ] );
         
         add_action( 'init', [ __CLASS__, 'handle_login_request' ] );
         add_action( 'init', [ __CLASS__, 'handle_frontend_actions' ] );
         add_action( 'wp_enqueue_scripts', [ __CLASS__, 'enqueue_frontend_assets' ] );
         add_action( 'template_redirect', [ __CLASS__, 'restrict_admin_access' ] );

         // Enqueue Login/Portal styles specifically if shortcode is present?
         // Optimally we enqueue globally or check for shortcode.
    }

    public static function restrict_admin_access() {
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) return;

        // Redirect non-admins trying to access wp-admin, unless doing something valid
        if ( is_admin() && ! current_user_can( 'manage_options' ) && ! defined('DOING_CRON') ) {
            wp_redirect( home_url( '/employee-portal/' ) );
            exit;
        }
    }

    public static function enqueue_frontend_assets() {
        // Enforce font-awesome or dashicons if not present?
        wp_enqueue_style( 'dashicons' );
        // Force cache bust to ensure CSS updates are applied immediately
        $version = time(); // defined('WP_DEBUG') && WP_DEBUG ? time() : '1.0.1'; 
        
        // Enqueue jsPDF library for PDF generation
        wp_enqueue_script( 'jspdf', 'https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js', [], '2.5.1', true );
        wp_enqueue_script( 'jspdf-autotable', 'https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.8.1/jspdf.plugin.autotable.min.js', ['jspdf'], '3.8.1', true );
        
        // Use HRM_PLUGIN_URL for correct path from root
        if ( defined('HRM_PLUGIN_URL') ) {
            wp_enqueue_style( 'hrm-frontend-css', HRM_PLUGIN_URL . 'assets/css/hrm-frontend.css', [], $version );
            wp_enqueue_style( 'hrm-icons-css', HRM_PLUGIN_URL . 'assets/css/hrm-icons.css', ['hrm-frontend-css'], $version );
            wp_enqueue_style( 'hrm-components-css', HRM_PLUGIN_URL . 'assets/css/hrm-components.css', ['hrm-icons-css'], $version );
            wp_enqueue_style( 'hrm-ui-polish-css', HRM_PLUGIN_URL . 'assets/css/hrm-ui-polish.css', ['hrm-components-css'], $version );
            wp_enqueue_style( 'hrm-responsive-fixes-css', HRM_PLUGIN_URL . 'assets/css/hrm-responsive-fixes.css', ['hrm-ui-polish-css'], $version );
            wp_enqueue_script( 'hrm-frontend-js', HRM_PLUGIN_URL . 'assets/js/hrm-frontend.js', ['jquery', 'jspdf', 'jspdf-autotable'], $version, true );
        }
        
        // Localize Script - NOTE: Key must be 'ajaxurl' (not 'ajax_url') to match hrm-frontend.js
        wp_localize_script( 'hrm-frontend-js', 'hrm_ajax', [
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'hrm_ajax_nonce' )
        ]);
    }

    public static function handle_login_request() {
        if ( isset( $_POST['hrm_login_submit'] ) && isset( $_POST['hrm_login_nonce'] ) ) {
            if ( ! wp_verify_nonce( $_POST['hrm_login_nonce'], 'hrm_login_action' ) ) return;

            $creds = array(
                'user_login'    => sanitize_text_field( $_POST['log'] ),
                'user_password' => $_POST['pwd'],
                'remember'      => isset( $_POST['rememberme'] ),
            );

            $user = wp_signon( $creds, false );

            if ( is_wp_error( $user ) ) {
                $redirect_url = add_query_arg( 'login_error', 'invalid', wp_get_referer() );
                wp_redirect( $redirect_url );
                exit;
            } else {
                if ( user_can( $user, 'manage_options' ) ) {
                    wp_redirect( admin_url() );
                } else {
                    wp_redirect( home_url( '/employee-portal/' ) );
                }
                exit;
            }
        }
    }

    public static function handle_frontend_actions() {
        if ( ! isset( $_POST['hrm_action'] ) ) return;
        
        $current_user = wp_get_current_user();
        if ( ! $current_user->ID ) return;
        
        // Find Employee ID (Robust)
        $employee_id = 0;
        
        // 1. Try by Author
        $employee_post = get_posts([
            'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
            'author'    => $current_user->ID,
            'numberposts' => 1,
            'fields'    => 'ids'
        ]);

        if ( ! empty( $employee_post ) ) {
             $employee_id = $employee_post[0];
        } else {
             // 2. Try by Email
             $employee_by_email = get_posts([
                'post_type'  => HRM_Setup::EMPLOYEE_POST_TYPE,
                'meta_key'   => 'hrm_email',
                'meta_value' => $current_user->user_email,
                'numberposts' => 1,
                'fields'     => 'ids'
            ]);
            if ( ! empty( $employee_by_email ) ) {
                $employee_id = $employee_by_email[0];
            }
        }
        
        if ( ! $employee_id ) return;

        // Delegate
        if ( $_POST['hrm_action'] === 'mark_attendance' ) {
            HRM_Attendance::handle_frontend_attendance( $employee_id );
        }
        if ( $_POST['hrm_action'] === 'update_profile' ) {
            HRM_Employees::handle_frontend_profile_update( $employee_id );
        }
        if ( $_POST['hrm_action'] === 'submit_progress' || $_POST['hrm_action'] === 'complete_task' ) {
            HRM_Tasks::handle_frontend_actions( $employee_id );
        }
    }

    public static function render_login_page() {
        // Get company logo (needs to be available for both logged-in and logged-out views)
        $logo_id = get_option( 'hrm_company_logo_id', 0 );
        $logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'medium' ) : '';
        
        if ( is_user_logged_in() ) {
            $current_user = wp_get_current_user();
            $redirect_url = home_url('/employee-portal/');
            $btn_text = 'Go to Employee Portal';

            if ( current_user_can( 'manage_options' ) ) {
                $redirect_url = admin_url();
                $btn_text = 'Go to Dashboard';
            }
            
            $logo_html = '';
            if ( $logo_url ) {
                $logo_html = '<div class="hrm-login-logo"><img src="' . esc_url( $logo_url ) . '" alt="' . esc_attr( get_bloginfo( 'name' ) ) . '" class="hrm-company-logo"></div>';
            }

            return '<div class="hrm-login-wrap logged-in">
                <div class="hrm-login-card">
                    ' . $logo_html . '
                    <h2>Welcome back, ' . esc_html( $current_user->display_name ) . '!</h2>
                    <p>You are already logged in.</p>
                    <a href="' . $redirect_url . '" class="hrm-btn-primary btn-block">' . $btn_text . '</a>
                </div>
            </div>';
        }

        $error = isset( $_GET['login_error'] ) ? $_GET['login_error'] : '';
        
        ob_start();
        ?>
        <div class="hrm-login-wrap">
            <div class="hrm-login-card">
                <?php if ( $logo_url ) : ?>
                    <div class="hrm-login-logo">
                        <img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>" class="hrm-company-logo">
                    </div>
                <?php endif; ?>
                
                <div class="hrm-login-header">
                    <h2>Employee Login</h2>
                    <p>Sign in to access your dashboard</p>
                </div>
                
                <?php if ( $error ) : ?>
                    <div class="hrm-login-error">
                        <?php echo $error === 'invalid' ? 'Invalid username or password.' : 'Login failed. Please try again.'; ?>
                    </div>
                <?php endif; ?>

                <form method="post" class="hrm-login-form">
                    <?php wp_nonce_field( 'hrm_login_action', 'hrm_login_nonce' ); ?>
                    <div class="hrm-form-group">
                        <label for="hrm_user_login">Username or Email</label>
                        <input type="text" name="log" id="hrm_user_login" class="input" value="" size="20" required>
                    </div>
                    <div class="hrm-form-group">
                        <label for="hrm_user_pass">Password</label>
                        <input type="password" name="pwd" id="hrm_user_pass" class="input" value="" size="20" required>
                    </div>
                    <div class="hrm-form-group hrm-remember-me">
                        <label><input name="rememberme" type="checkbox" id="rememberme" value="forever"> Remember Me</label>
                    </div>
                    <div class="hrm-form-actions">
                        <button type="submit" name="wp-submit" class="hrm-btn-primary btn-block">Sign In</button>
                        <input type="hidden" name="hrm_login_submit" value="1">
                    </div>
                </form>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public static function render_employee_portal() {
        if ( ! is_user_logged_in() ) {
            return '<p>Please <a href="'.home_url('/employee-login/').'">log in</a> to access the employee portal.</p>';
        }
    
        $current_user = wp_get_current_user();
        if ( current_user_can( 'manage_options' ) ) {
            // Admin viewing portal?
            // Allow them to see it, but handle "No employee record" case.
        }

        // Find Employee (Robust)
        $employee_id = 0;
        
        // 1. Try by Author
        $employee = get_posts([
            'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
            'author'    => $current_user->ID,
            'numberposts' => 1
        ]);
        
        if ( ! empty($employee) ) {
            $employee_id = $employee[0]->ID;
        } else {
            // 2. Try by Email
             $employee_by_email = get_posts([
                'post_type'  => HRM_Setup::EMPLOYEE_POST_TYPE,
                'meta_key'   => 'hrm_email',
                'meta_value' => $current_user->user_email,
                'numberposts' => 1
            ]);
            if ( ! empty( $employee_by_email ) ) {
                $employee_id = $employee_by_email[0]->ID;
            }
        }
    
        if ( ! $employee_id ) {
            return '<p>No employee record found for your account. Please contact HR.</p>';
        }
    
        // Enqueue Assets again just in case Shortcode is used outside dedicated page
        wp_enqueue_script('hrm-frontend-js');
        wp_enqueue_style('hrm-frontend-css');
        
        // Get company logo
        $logo_id = get_option( 'hrm_company_logo_id', 0 );
        $logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'medium' ) : '';

        ob_start();
        ?>
        <div class="hrm-portal-wrap">
            <!-- Mobile Toggle -->
            <button class="hrm-menu-toggle">
                <span class="dashicons dashicons-menu"></span>
            </button>
    
            <!-- Sidebar -->
            <div class="hrm-sidebar">
                <div class="hrm-sidebar-header">
                    <?php if ( $logo_url ) : ?>
                        <div class="hrm-portal-logo">
                            <img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>" class="hrm-company-logo">
                        </div>
                    <?php else : ?>
                        <h2><span class="dashicons dashicons-groups" style="font-size: 28px; width: 28px; height: 28px; vertical-align: middle;"></span> HRM Portal</h2>
                    <?php endif; ?>
                </div>
                
                <div class="hrm-nav-scroll">
                    <!-- Main Section -->
                    <div class="hrm-nav-section">
                        <h4 class="hrm-nav-label">Main</h4>
                        <ul class="hrm-nav-menu">
                            <li class="hrm-tab-btn active" onclick="openTab(event, 'dashboard')">
                                <span class="dashicons dashicons-dashboard"></span> Dashboard
                            </li>
                        </ul>
                    </div>

                    <!-- Work Section -->
                    <div class="hrm-nav-section">
                        <h4 class="hrm-nav-label">Work</h4>
                        <ul class="hrm-nav-menu">
                            <li class="hrm-tab-btn" onclick="openTab(event, 'tasks')">
                                <span class="dashicons dashicons-list-view"></span> My Tasks
                            </li>
                            <li class="hrm-tab-btn" onclick="openTab(event, 'daily_progress')">
                                <span class="dashicons dashicons-chart-bar"></span> Daily Progress
                            </li>
                            <li class="hrm-tab-btn" onclick="openTab(event, 'leave')">
                                <span class="dashicons dashicons-calendar-alt"></span> Leave Requests
                            </li>
                            <li class="hrm-tab-btn" onclick="openTab(event, 'policies')">
                                <span class="dashicons dashicons-media-document"></span> Company Policies
                            </li>
                            <li class="hrm-tab-btn" onclick="openTab(event, 'payroll')">
                                <span class="dashicons dashicons-money-alt"></span> My Payroll
                            </li>
                        </ul>
                    </div>

                    <!-- Account Section -->
                    <div class="hrm-nav-section">
                        <h4 class="hrm-nav-label">Account</h4>
                        <ul class="hrm-nav-menu">
                            <li class="hrm-tab-btn" onclick="openTab(event, 'notifications')">
                                <span class="dashicons dashicons-bell"></span> Notifications
                                <span class="nav-notif-badge"></span>
                            </li>
                            <li class="hrm-tab-btn" onclick="openTab(event, 'profile')">
                                <span class="dashicons dashicons-id"></span> My Profile
                            </li>
                            <li class="hrm-tab-btn logout">
                                <a href="<?php echo wp_logout_url( home_url() ); ?>">
                                    <span class="dashicons dashicons-exit"></span> Logout
                                </a>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
    
            <!-- Content Area -->
            <div class="hrm-content-area">
                <!-- Sticky Header -->
                <!-- Sticky Header -->
                <div class="hrm-top-bar">
                    <div class="hrm-search-container">
                        <button class="hrm-search-mobile-toggle" id="hrm-mobile-search-trigger">
                            <span class="dashicons dashicons-search"></span>
                        </button>
                        <div class="hrm-search-wrap" id="hrm-search-wrapper">
                             <span class="dashicons dashicons-search"></span>
                             <input type="text" placeholder="Search tasks, files, policies..." class="hrm-top-search" id="hrm-portal-search-input">
                             <button class="hrm-search-close-mobile" id="hrm-mobile-search-close">
                                 <span class="dashicons dashicons-no-alt"></span>
                             </button>
                        </div>
                    </div>
                    <div class="hrm-top-actions">
                        <div class="hrm-notif-bell" title="Notifications">
                            <span class="dashicons dashicons-bell"></span>
                            <span class="notif-badge"></span>
                        </div>
                        <div class="hrm-user-pill" onclick="openTab(event, 'profile')">
                             <div class="user-avatar-small">
                                 <?php echo strtoupper(substr($current_user->display_name, 0, 1)); ?>
                             </div>
                             <span class="user-name"><?php echo esc_html($current_user->display_name); ?></span>
                        </div>
                    </div>
                </div>

                <!-- Notifications Toast Container -->
                <?php HRM_Notifications::render_frontend_notifications( $employee_id ); ?>
                
                <div class="hrm-main-scroll">
                    <!-- Dashboard Tab -->
                    <div id="dashboard" class="hrm-tab-content active">
                        <div class="hrm-hero-section">
                            <div class="hero-content">
                                <h1>Good <?php echo (date('H') < 12) ? 'Morning' : ((date('H') < 17) ? 'Afternoon' : 'Evening'); ?>, <?php echo esc_html( explode(' ', $current_user->display_name)[0] ); ?>!</h1>
                                <p><?php echo date('l, F j, Y'); ?></p>
                            </div>
                        </div>
                        
                        <?php 
                        // --- Dashboard Stats Logic ---
                        global $wpdb;
                        $month_start = date('Y-m-01');
                        
                        // 1. Pending Tasks
                        $pending_count = $wpdb->get_var( $wpdb->prepare( 
                            "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_tasks WHERE assigned_to = %d AND status = 'pending'", 
                            $employee_id 
                        ));


                        // 2. Monthly Hours
                        $monthly_hours = $wpdb->get_var( $wpdb->prepare( 
                            "SELECT SUM(hours_worked) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date >= %s", 
                            $employee_id, $month_start 
                        ));
                        $monthly_hours = $monthly_hours ? round($monthly_hours, 1) : 0;

                        // 3. Attendance Rate (This Month)
                        $total_records = $wpdb->get_var( $wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date >= %s",
                            $employee_id, $month_start
                        ));
                        $present_cnt = $wpdb->get_var( $wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date >= %s AND status = 'present'",
                            $employee_id, $month_start
                        ));
                         $attendance_rate = ($total_records > 0) ? round(($present_cnt / $total_records) * 100) : 100;
                        ?>

                        <div class="hrm-stats-row">
                            <div class="hrm-mini-stat" onclick="openTab(event, 'tasks')" style="cursor: pointer;">
                                <div class="stat-header">
                                    <span class="label">Pending Tasks</span>
                                    <span class="dashicons dashicons-list-view icon"></span>
                                </div>
                                <div class="value"><?php echo $pending_count; ?></div>
                            </div>
                            <div class="hrm-mini-stat" onclick="openTab(event, 'daily_progress')" style="cursor: pointer;">
                                <div class="stat-header">
                                    <span class="label">Hours (<?php echo date('M'); ?>)</span>
                                    <span class="dashicons dashicons-clock icon" style="color:#10B981;"></span>
                                </div>
                                <div class="value"><?php echo $monthly_hours; ?></div>
                            </div>
                            <div class="hrm-mini-stat" onclick="document.querySelector('.hrm-attendance-box').scrollIntoView({behavior: 'smooth'})" style="cursor: pointer;">
                                <div class="stat-header">
                                    <span class="label">Attendance %</span>
                                    <span class="dashicons dashicons-chart-area icon" style="color:#3B82F6;"></span>
                                </div>
                                <div class="value"><?php echo $attendance_rate; ?>%</div>
                            </div>
                        </div>
                        
                        <!-- Attendance -->
                        <?php HRM_Attendance::render_frontend_attendance( $employee_id ); ?>
                        
                        <!-- Notifications List -->
                        <div style="margin-top: 20px;">
                            <?php HRM_Notifications::render_frontend_notifications_list( $employee_id ); ?>
                        </div>
                    </div>
        
                    <!-- Tasks Tab -->
                    <div id="tasks" class="hrm-tab-content">
                        <?php HRM_Tasks::render_frontend_tasks( $employee_id ); ?>
                    </div>

                    <!-- Daily Progress Tab -->
                    <div id="daily_progress" class="hrm-tab-content">
                        <?php HRM_Tasks::render_frontend_progress( $employee_id ); ?>
                    </div>

                    <!-- Leave Tab -->
                    <div id="leave" class="hrm-tab-content">
                        <?php HRM_Leave::render_employee_leave(); ?>
                    </div>

                    <!-- Company Policies Tab -->
                    <div id="policies" class="hrm-tab-content">
                        <?php HRM_Policies::render_employee_policies(); ?>
                    </div>

                    <!-- Payroll Tab -->
                    <div id="payroll" class="hrm-tab-content">
                        <?php HRM_Payroll::render_employee_payroll( $employee_id ); ?>
                    </div>
        
                    <!-- Profile Tab -->
                    <div id="profile" class="hrm-tab-content">
                        <?php HRM_Employees::render_frontend_profile( $employee_id ); ?>
                    </div>
                    
                    <!-- Notifications Tab -->
                    <div id="notifications" class="hrm-tab-content">
                        <?php HRM_Notifications::render_notifications_page( $employee_id ); ?>
                    </div>
                </div>
            </div>
            
             <!-- Mobile Overlay -->
            <div class="hrm-overlay" onclick="document.querySelector('.hrm-sidebar').classList.remove('active')"></div>
        </div>
        
        <script>
        function openTab(evt, tabName) {
            if(!tabName || tabName === 'logout') return;
            var i, tabcontent, tablinks;
            
            // Hide all contents
            tabcontent = document.getElementsByClassName("hrm-tab-content");
            for (i = 0; i < tabcontent.length; i++) {
                tabcontent[i].style.display = "none";
                tabcontent[i].classList.remove("active");
            }
            
            // Remove active class from links
            tablinks = document.getElementsByClassName("hrm-tab-btn");
            for (i = 0; i < tablinks.length; i++) {
                tablinks[i].classList.remove("active");
            }
            
            // Show current tab
            var targetTab = document.getElementById(tabName);
            if (targetTab) {
                targetTab.style.display = "block";
                // Animation frame for smooth transition
                requestAnimationFrame(function() {
                    targetTab.classList.add("active");
                });
            }
            
            // Update active state on button
            if (evt && evt.currentTarget) {
                evt.currentTarget.classList.add("active");
            } else {
                // If triggered programmatically without event, find the button
                var btn = document.querySelector('.hrm-tab-btn[onclick*="' + tabName + '"]');
                if (btn) btn.classList.add("active");
            }
            
            // Mobile: Close sidebar
            var sidebar = document.querySelector('.hrm-sidebar');
            if (sidebar) {
                sidebar.classList.remove('active');
            }
            
            // Restore body overflow if mobile sidebar was open
            document.body.style.overflow = '';
        }
        </script>
        <?php
        return ob_get_clean();
    }
}
